<?php
/*
Plugin Name: DWD Masonry Posts
Plugin URI: https://detailwebdesign.com/plugins/dwd-masonry-posts
Description: Display WordPress posts in a true masonry grid layout. Customize the number of posts, categories, columns, gap size, overlay color, title color, and meta color directly from the settings panel. Fully responsive and mobile-friendly. Shortcodes: [dwd_masonry] to display posts, [dwd_masonry include="1,2,3"] to display specific posts by ID.
Version: 1.0.0
Author: DetailWebDesign
Author URI: https://detailwebdesign.com
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: dwd-masonry-posts
Domain Path: /languages
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/* Admin menu & settings */
add_action( 'admin_menu', 'dwd_masonry_add_admin_menu' );
add_action( 'admin_init', 'dwd_masonry_settings_init' );

function dwd_masonry_add_admin_menu() {
    add_options_page(
        __('DWD Masonry Posts', 'dwd-masonry-posts'),
        __('DWD Masonry Posts', 'dwd-masonry-posts'),
        'manage_options',
        'dwd-masonry-posts',
        'dwd_masonry_options_page'
    );
}

function dwd_masonry_settings_init() {
    register_setting( 'dwdMasonryGroup', 'dwd_masonry_settings', array( 'sanitize_callback' => 'dwd_masonry_sanitize_options' ) );

    add_settings_section(
        'dwd_masonry_section',
        __('Masonry Layout Settings', 'dwd-masonry-posts'),
        null,
        'dwdMasonryGroup'
    );

    add_settings_field(
        'dwd_masonry_post_count',
        __('Number of Posts to Display', 'dwd-masonry-posts'),
        'dwd_masonry_post_count_render',
        'dwdMasonryGroup',
        'dwd_masonry_section'
    );

    add_settings_field(
        'dwd_masonry_categories',
        __('Categories', 'dwd-masonry-posts'),
        'dwd_masonry_categories_render',
        'dwdMasonryGroup',
        'dwd_masonry_section'
    );

    add_settings_field(
        'dwd_masonry_columns',
        __('Number of Columns', 'dwd-masonry-posts'),
        'dwd_masonry_columns_render',
        'dwdMasonryGroup',
        'dwd_masonry_section'
    );

    add_settings_field(
        'dwd_masonry_gap',
        __('Gap Size', 'dwd-masonry-posts'),
        'dwd_masonry_gap_render',
        'dwdMasonryGroup',
        'dwd_masonry_section'
    );

    add_settings_field(
        'dwd_masonry_overlay_color',
        __('Overlay Background Color', 'dwd-masonry-posts'),
        'dwd_masonry_overlay_color_render',
        'dwdMasonryGroup',
        'dwd_masonry_section'
    );

    add_settings_field(
        'dwd_masonry_overlay_opacity',
        __('Overlay Opacity (0-1)', 'dwd-masonry-posts'),
        'dwd_masonry_overlay_opacity_render',
        'dwdMasonryGroup',
        'dwd_masonry_section'
    );

    add_settings_field(
        'dwd_masonry_title_color',
        __('Title (h6) Color', 'dwd-masonry-posts'),
        'dwd_masonry_title_color_render',
        'dwdMasonryGroup',
        'dwd_masonry_section'
    );

    add_settings_field(
        'dwd_masonry_meta_color',
        __('Meta (Date/Category) Color', 'dwd-masonry-posts'),
        'dwd_masonry_meta_color_render',
        'dwdMasonryGroup',
        'dwd_masonry_section'
    );
}

function dwd_masonry_sanitize_options( $input ) {
    $out = array();
    $out['dwd_masonry_post_count'] = isset($input['dwd_masonry_post_count']) ? max(1, intval($input['dwd_masonry_post_count'])) : 10;

    if ( isset($input['dwd_masonry_categories']) && is_array($input['dwd_masonry_categories']) ) {
        if ( in_array('all', $input['dwd_masonry_categories'], true) ) {
            $out['dwd_masonry_categories'] = array('all');
        } else {
            $out['dwd_masonry_categories'] = array_map('intval', $input['dwd_masonry_categories']);
        }
    } else {
        $out['dwd_masonry_categories'] = array('all');
    }

    $columns_allowed = array(3,4,5,6);
    $col = isset($input['dwd_masonry_columns']) ? intval($input['dwd_masonry_columns']) : 4;
    $out['dwd_masonry_columns'] = in_array($col, $columns_allowed, true) ? $col : 4;

    $gap_allowed = array(1,2,3);
    $gap = isset($input['dwd_masonry_gap']) ? intval($input['dwd_masonry_gap']) : 2;
    $out['dwd_masonry_gap'] = in_array($gap, $gap_allowed, true) ? $gap : 2;

    $out['dwd_masonry_overlay_color'] = sanitize_hex_color( $input['dwd_masonry_overlay_color'] ?? '#000000' );
    $out['dwd_masonry_overlay_opacity'] = min(1, max(0, floatval($input['dwd_masonry_overlay_opacity'] ?? 0.55)));
    $out['dwd_masonry_title_color'] = sanitize_hex_color( $input['dwd_masonry_title_color'] ?? '#ffffff' );
    $out['dwd_masonry_meta_color'] = sanitize_hex_color( $input['dwd_masonry_meta_color'] ?? '#cccccc' );

    return $out;
}

/* Render callbacks */
function dwd_masonry_post_count_render() {
    $options = get_option('dwd_masonry_settings');
    $val = esc_attr( $options['dwd_masonry_post_count'] ?? 10 );
    echo '<input type="number" name="dwd_masonry_settings[dwd_masonry_post_count]" value="' . esc_attr($val) . '" min="1" style="width:90px;">';
}

function dwd_masonry_categories_render() {
    $options = get_option('dwd_masonry_settings');
    $selected = $options['dwd_masonry_categories'] ?? array('all');
    $categories = get_categories(array('hide_empty' => false));
    echo '<select name="dwd_masonry_settings[dwd_masonry_categories][]" multiple size="8" style="width:320px;">';
    $sel_all = in_array('all', (array)$selected, true) ? 'selected' : '';
    echo '<option value="all" ' . esc_attr($sel_all) . '>'. esc_html__( 'All', 'dwd-masonry-posts' ) .'</option>';
    foreach( $categories as $cat ) {
        $sel = in_array($cat->term_id, (array)$selected, true) ? 'selected' : '';
        echo '<option value="'. intval($cat->term_id) .'" ' . esc_attr($sel) . '>'. esc_html($cat->name) .'</option>';
    }
    echo '</select>';
}

function dwd_masonry_columns_render() {
    $options = get_option('dwd_masonry_settings');
    $val = intval( $options['dwd_masonry_columns'] ?? 4 );
    echo '<select name="dwd_masonry_settings[dwd_masonry_columns]">';
    foreach( array(3,4,5,6) as $c ) {
        $s = ($c === $val) ? 'selected' : '';
        echo '<option value="' . esc_attr($c) . '" ' . esc_attr($s) . '>' . esc_html($c) . ' '. esc_html__( 'columns', 'dwd-masonry-posts' ) .'</option>';
    }
    echo '</select>';
}

function dwd_masonry_gap_render() {
    $options = get_option('dwd_masonry_settings');
    $val = intval( $options['dwd_masonry_gap'] ?? 2 );
    echo '<select name="dwd_masonry_settings[dwd_masonry_gap]">';
    foreach( array(1,2,3) as $g ) {
        $s = ($g === $val) ? 'selected' : '';
        $label = $g . ' (' . ($g*5) . 'px)';
        echo '<option value="' . esc_attr($g) . '" ' . esc_attr($s) . '>'. esc_html($label) .'</option>';
    }
    echo '</select>';
}

function dwd_masonry_overlay_color_render() {
    $options = get_option('dwd_masonry_settings');
    $val = esc_attr( $options['dwd_masonry_overlay_color'] ?? '#000000' );
    echo '<input type="color" name="dwd_masonry_settings[dwd_masonry_overlay_color]" value="' . esc_attr($val) . '">';
}

function dwd_masonry_overlay_opacity_render() {
    $options = get_option('dwd_masonry_settings');
    $val = esc_attr( $options['dwd_masonry_overlay_opacity'] ?? 0.55 );
    echo '<input type="number" step="0.05" min="0" max="1" name="dwd_masonry_settings[dwd_masonry_overlay_opacity]" value="' . esc_attr($val) . '" style="width:90px;">';
}

function dwd_masonry_title_color_render() {
    $options = get_option('dwd_masonry_settings');
    $val = esc_attr( $options['dwd_masonry_title_color'] ?? '#ffffff' );
    echo '<input type="color" name="dwd_masonry_settings[dwd_masonry_title_color]" value="' . esc_attr($val) . '">';
}

function dwd_masonry_meta_color_render() {
    $options = get_option('dwd_masonry_settings');
    $val = esc_attr( $options['dwd_masonry_meta_color'] ?? '#cccccc' );
    echo '<input type="color" name="dwd_masonry_settings[dwd_masonry_meta_color]" value="' . esc_attr($val) . '">';
}

function dwd_masonry_options_page() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html__( 'DWD Masonry Posts', 'dwd-masonry-posts' ); ?></h1>
        <div class="dwd-admin-card">
            <form action="options.php" method="post">
                <?php
                settings_fields('dwdMasonryGroup');
                do_settings_sections('dwdMasonryGroup');
                submit_button();
                ?>
            </form>
        </div>
    </div>
    <?php
}

/* Public CSS & JS enqueue */
add_action( 'wp_enqueue_scripts', 'dwd_masonry_enqueue_public_assets' );
function dwd_masonry_enqueue_public_assets() {
    $options = get_option('dwd_masonry_settings');
    $gap_choice = intval( $options['dwd_masonry_gap'] ?? 2 );
    $gap_px = $gap_choice * 5;

    wp_enqueue_style(
        'dwd-masonry-posts',
        plugin_dir_url( __FILE__ ) . 'public/css/dwd-masonry-posts.css',
        array(),
        '1.0.0'
    );

    wp_enqueue_script(
        'dwd-masonry-posts',
        plugin_dir_url( __FILE__ ) . 'public/js/dwd-masonry-posts.js',
        array(),
        '1.0.0',
        true
    );

    wp_localize_script( 'dwd-masonry-posts', 'dwdMasonryVars', array(
        'gap_px' => $gap_px,
    ));
}

/* Admin CSS enqueue */
add_action( 'admin_enqueue_scripts', 'dwd_masonry_enqueue_admin_assets' );
function dwd_masonry_enqueue_admin_assets( $hook ) {
    if ( $hook === 'settings_page_dwd-masonry-posts' ) {
        wp_enqueue_style(
            'dwd-masonry-admin',
            plugin_dir_url( __FILE__ ) . 'admin/css/dwd-masonry-admin.css',
            array(),
            '1.0.0'
        );
    }
}

/* Shortcode */
add_shortcode( 'dwd_masonry', 'dwd_masonry_shortcode' );
function dwd_masonry_shortcode( $atts ) {
    // Viewport meta tag kontrolü ve eklenmesi
    add_action('wp_head', function() {
        echo '<meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">' . "\n";
    }, 1);

    $atts = shortcode_atts( array(
        'include' => ''
    ), $atts, 'dwd_masonry' );

    $options = get_option('dwd_masonry_settings');

    $post_count = intval( $options['dwd_masonry_post_count'] ?? 10 );
    $selected_cats = $options['dwd_masonry_categories'] ?? array('all');
    $columns = intval( $options['dwd_masonry_columns'] ?? 4 );
    $gap_choice = intval( $options['dwd_masonry_gap'] ?? 2 );
    $gap_px = $gap_choice * 5;

    $overlay_color = esc_attr( $options['dwd_masonry_overlay_color'] ?? '#000000' );
    $overlay_opacity = floatval( $options['dwd_masonry_overlay_opacity'] ?? 0.55 );
    $title_color = esc_attr( $options['dwd_masonry_title_color'] ?? '#ffffff' );
    $meta_color = esc_attr( $options['dwd_masonry_meta_color'] ?? '#cccccc' );

    $args = array(
        'post_type' => 'post',
        'posts_per_page' => $post_count,
        'orderby' => 'date',
        'order' => 'DESC'
    );

    if ( ! empty( $atts['include'] ) ) {
        $ids = array_filter( array_map('intval', explode(',', $atts['include'])) );
        if ( ! empty( $ids ) ) {
            $args['post__in'] = $ids;
            $args['orderby'] = 'post__in';
            $args['posts_per_page'] = count($ids);
        }
    } elseif ( ! empty( $selected_cats ) && ! in_array('all', $selected_cats, true) ) {
        $args['category__in'] = array_map('intval', $selected_cats);
    }

    $query = new WP_Query( $args );

    ob_start();
    ?>
<style>
.dwd-grid-container{display:grid;grid-template-columns: repeat(<?php echo esc_attr($columns); ?>, 1fr);grid-gap: <?php echo esc_attr($gap_px); ?>px;}
.dwd-grid-item .dwd-overlay{position:absolute;left:0;right:0;bottom:0;padding:12px;background: <?php echo esc_attr($overlay_color); ?>;opacity: <?php echo esc_attr($overlay_opacity); ?>;color:<?php echo esc_attr($title_color); ?>;}
.dwd-grid-item h6{color:<?php echo esc_attr($title_color); ?>;margin:0 0 6px 0;font-size:15px;line-height:1.4;}
.dwd-grid-item .excerpt{font-size:11px;line-height:1.5;color:<?php echo esc_attr($meta_color); ?>;margin-bottom:6px;max-height:3.2em;overflow:hidden;text-overflow:ellipsis;display:-webkit-box;-webkit-line-clamp:1;-webkit-box-orient: vertical;}
.dwd-grid-item .meta{color:<?php echo esc_attr($meta_color); ?>;font-size:12px;margin-top:4px;display:flex;gap:8px;align-items:center;}
@media screen and (max-width: 1024px){.dwd-grid-item{margin: 0 auto <?php echo esc_attr($gap_px); ?>px auto !important;}.dwd-grid-item .dwd-overlay{background: <?php echo esc_attr($overlay_color); ?> !important;opacity: <?php echo esc_attr($overlay_opacity); ?> !important;}.dwd-grid-item h6{color: <?php echo esc_attr($title_color); ?> !important;}.dwd-grid-item .meta{color: <?php echo esc_attr($meta_color); ?> !important;}}
<?php $gap_mobile = max(8, intval($gap_px)); ?>
@media screen and (max-width: 480px){.dwd-grid-item{margin: 0 0 <?php echo esc_attr($gap_mobile); ?>px 0 !important;}}
@media screen and (min-width: 1025px){.dwd-grid-container{grid-template-columns: repeat(<?php echo esc_attr($columns); ?>, 1fr) !important;grid-gap: <?php echo esc_attr($gap_px); ?>px !important;}}
</style>

    <div class="dwd-grid-container" role="list">
        <?php
// Dinamik pattern setleri sütun sayısına göre
switch ($columns) {
    case 3:
        $patterns = array(
            array('c'=>1,'r'=>1),
            array('c'=>2,'r'=>1),
            array('c'=>1,'r'=>2),
            array('c'=>3,'r'=>1),
            array('c'=>2,'r'=>2),
            array('c'=>1,'r'=>1),
            array('c'=>2,'r'=>1),
            array('c'=>1,'r'=>3),
            array('c'=>3,'r'=>2),
            array('c'=>1,'r'=>1)
        );
        break;

    case 4:
        $patterns = array(
            array('c'=>2,'r'=>1),
            array('c'=>1,'r'=>2),
            array('c'=>1,'r'=>1),
            array('c'=>2,'r'=>2),
            array('c'=>1,'r'=>1),
            array('c'=>3,'r'=>1),
            array('c'=>1,'r'=>3),
            array('c'=>2,'r'=>1),
            array('c'=>1,'r'=>2),
            array('c'=>2,'r'=>2),
            array('c'=>1,'r'=>1),
            array('c'=>4,'r'=>1),
            array('c'=>2,'r'=>3),
            array('c'=>1,'r'=>1)
        );
        break;

    case 5:
        $patterns = array(
            array('c'=>2,'r'=>1),
            array('c'=>1,'r'=>2),
            array('c'=>3,'r'=>1),
            array('c'=>2,'r'=>2),
            array('c'=>1,'r'=>1),
            array('c'=>4,'r'=>2),
            array('c'=>1,'r'=>3),
            array('c'=>2,'r'=>1),
            array('c'=>3,'r'=>2),
            array('c'=>1,'r'=>1),
            array('c'=>5,'r'=>1),
            array('c'=>2,'r'=>2),
            array('c'=>1,'r'=>1),
            array('c'=>2,'r'=>3)
        );
        break;

    case 6:
    default:
        $patterns = array(
            array('c'=>2,'r'=>1),
            array('c'=>1,'r'=>2),
            array('c'=>3,'r'=>1),
            array('c'=>2,'r'=>2),
            array('c'=>1,'r'=>1),
            array('c'=>2,'r'=>3),
            array('c'=>1,'r'=>1),
            array('c'=>4,'r'=>2),
            array('c'=>2,'r'=>1),
            array('c'=>1,'r'=>2),
            array('c'=>2,'r'=>2),
            array('c'=>6,'r'=>1),
            array('c'=>3,'r'=>2),
            array('c'=>1,'r'=>1),
            array('c'=>2,'r'=>1),
            array('c'=>2,'r'=>2),
            array('c'=>1,'r'=>3)
        );
        break;
}

        $i = 0;
        if ( $query->have_posts() ) :
            while ( $query->have_posts() ): $query->the_post();
                $bg = has_post_thumbnail() ? get_the_post_thumbnail_url(get_the_ID(), 'large') : '';
                $dim = $patterns[$i % count($patterns)];
                $categories = get_the_category();
                $cat_name = !empty($categories) ? $categories[0]->name : '';
                ?>
                <div class="dwd-grid-item"
                     style="grid-column: span <?php echo esc_attr($dim['c']); ?>;
                            grid-row: span <?php echo esc_attr($dim['r']); ?>;
                            <?php if($bg) echo 'background-image:url(' . esc_url($bg) . ');'; ?>">
                    <a href="<?php the_permalink(); ?>">
                        <div class="dwd-overlay" aria-hidden="false">
                            <h6><?php the_title(); ?></h6>

                            <div class="excerpt">
                                <?php echo esc_html( wp_trim_words( get_the_excerpt(), 25, '...' ) ); ?>
                            </div>

                            <div class="meta">
                                <span class="date"><svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" viewBox="0 0 24 24" aria-hidden="true"><path d="M7 2v2H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2h-2V2h-2v2H9V2H7zm12 18H5V10h14v10z"/></svg> <?php echo esc_html( get_the_date() ); ?></span>
                                <?php if($cat_name): ?>
                                    <span class="category">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" viewBox="0 0 24 24" aria-hidden="true"><path d="M10 4H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V8c0-1.1-.9-2-2-2h-8l-2-2z"/></svg> <?php echo esc_html($cat_name); ?></span>
                                <?php endif; ?>
                            </div>

                        </div>
                    </a>
                </div>
                <?php
                $i++;
            endwhile;
            wp_reset_postdata();
        else:
            echo '<div style="padding:18px;">'. esc_html__( 'No posts available.', 'dwd-masonry-posts' ) .'</div>';
        endif;
        ?>
    </div>
    <?php
    return ob_get_clean();
}